<?php
/*
Plugin Name: Museumkaart Breakeven Checker
Description: Bereken snel na hoeveel museumbezoeken de Museumkaart is terugverdiend (incl. volwassenen/kinderen).
Version: 1.1.1
Author: Alexander van Dijl
License: GPL2+
*/

if (!defined('ABSPATH')) exit;

define('MBC_VER', '1.1.1');

function mbc_default_options() {
    return array(
        'adult_price' => 79.00,
        'child_price' => 39.00,
        'default_ticket_price' => 17.50,
        'default_adults' => 1,
        'default_children' => 0,
    );
}

function mbc_get_options() {
    $defaults = mbc_default_options();
    $opts = get_option('mbc_options', array());
    if (!is_array($opts)) $opts = array();
    return array_merge($defaults, $opts);
}

add_action('admin_menu', function() {
    add_options_page(
        'Museumkaart Breakeven',
        'Museumkaart Breakeven',
        'manage_options',
        'mbc-settings',
        'mbc_render_settings_page'
    );
});

add_action('admin_init', function() {
    register_setting('mbc_settings_group', 'mbc_options', 'mbc_sanitize_options');
});

function mbc_sanitize_options($input) {
    $defaults = mbc_default_options();
    $out = array();

    $out['adult_price'] = isset($input['adult_price']) ? floatval(str_replace(',', '.', $input['adult_price'])) : $defaults['adult_price'];
    $out['child_price'] = isset($input['child_price']) ? floatval(str_replace(',', '.', $input['child_price'])) : $defaults['child_price'];
    $out['default_ticket_price'] = isset($input['default_ticket_price']) ? floatval(str_replace(',', '.', $input['default_ticket_price'])) : $defaults['default_ticket_price'];

    $out['default_adults'] = isset($input['default_adults']) ? max(0, intval($input['default_adults'])) : $defaults['default_adults'];
    $out['default_children'] = isset($input['default_children']) ? max(0, intval($input['default_children'])) : $defaults['default_children'];

    if ($out['adult_price'] < 0) $out['adult_price'] = $defaults['adult_price'];
    if ($out['child_price'] < 0) $out['child_price'] = $defaults['child_price'];
    if ($out['default_ticket_price'] < 0) $out['default_ticket_price'] = $defaults['default_ticket_price'];

    return $out;
}

function mbc_render_settings_page() {
    if (!current_user_can('manage_options')) return;
    $opts = mbc_get_options();
    ?>
    <div class="wrap">
        <h1>Museumkaart Breakeven</h1>
        <p>Stel hier de standaardwaarden in voor de calculator-shortcode <code>[museumkaart_breakeven]</code>.</p>

        <form method="post" action="options.php">
            <?php settings_fields('mbc_settings_group'); ?>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="mbc_adult_price">Prijs Museumkaart (volwassene)</label></th>
                    <td>
                        <input name="mbc_options[adult_price]" id="mbc_adult_price" type="text"
                               value="<?php echo esc_attr(number_format((float)$opts['adult_price'], 2, ',', '.')); ?>"
                               class="regular-text" />
                        <p class="description">Bijv. 79,00</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="mbc_child_price">Prijs Museumkaart (kind)</label></th>
                    <td>
                        <input name="mbc_options[child_price]" id="mbc_child_price" type="text"
                               value="<?php echo esc_attr(number_format((float)$opts['child_price'], 2, ',', '.')); ?>"
                               class="regular-text" />
                        <p class="description">Bijv. 39,00</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="mbc_ticket_price">Standaard ticketprijs (per persoon per bezoek)</label></th>
                    <td>
                        <input name="mbc_options[default_ticket_price]" id="mbc_ticket_price" type="text"
                               value="<?php echo esc_attr(number_format((float)$opts['default_ticket_price'], 2, ',', '.')); ?>"
                               class="regular-text" />
                        <p class="description">Bijv. 17,50</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="mbc_default_adults">Standaard aantal volwassenen</label></th>
                    <td><input name="mbc_options[default_adults]" id="mbc_default_adults" type="number" min="0"
                               value="<?php echo esc_attr((int)$opts['default_adults']); ?>" class="small-text" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="mbc_default_children">Standaard aantal kinderen</label></th>
                    <td><input name="mbc_options[default_children]" id="mbc_default_children" type="number" min="0"
                               value="<?php echo esc_attr((int)$opts['default_children']); ?>" class="small-text" /></td>
                </tr>
            </table>

            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

function mbc_enqueue_assets() {
    wp_enqueue_style('mbc-style', plugins_url('assets/mbc.css', __FILE__), array(), MBC_VER);
    wp_enqueue_script('mbc-script', plugins_url('assets/mbc.js', __FILE__), array(), MBC_VER, true);

    $opts = mbc_get_options();
    wp_localize_script('mbc-script', 'MBC', array(
        'adult_price' => (float)$opts['adult_price'],
        'child_price' => (float)$opts['child_price'],
        'default_ticket_price' => (float)$opts['default_ticket_price'],
        'default_adults' => (int)$opts['default_adults'],
        'default_children' => (int)$opts['default_children'],
        'i18n' => array(
            'invalid' => 'Vul geldige waarden in (ticketprijs > 0 en minimaal 1 persoon).',
            'months_hint' => 'Vul bezoeken per maand in voor een indicatie in maanden.',
        ),
    ));
}

function mbc_shortcode($atts) {
    mbc_enqueue_assets();

    $id = 'mbc_' . wp_generate_uuid4();
    return '
    <div class="mbc" id="'.esc_attr($id).'" data-mbc>
        <div class="mbc-card">
            <div class="mbc-title">🎫 Museumkaart breakeven berekenen</div>
            <div class="mbc-subtitle">Bereken in 10 seconden na hoeveel bezoeken de Museumkaart is terugverdiend.</div>

            <div class="mbc-grid">
                <div class="mbc-field">
                    <label>Prijs Museumkaart (volwassene)</label>
                    <input type="text" inputmode="decimal" class="mbc-adult-price" value="" placeholder="79,00" />
                </div>
                <div class="mbc-field">
                    <label>Prijs Museumkaart (kind)</label>
                    <input type="text" inputmode="decimal" class="mbc-child-price" value="" placeholder="39,00" />
                </div>

                <div class="mbc-field">
                    <label>Aantal volwassenen</label>
                    <input type="number" min="0" class="mbc-adults" value="" />
                </div>
                <div class="mbc-field">
                    <label>Aantal kinderen</label>
                    <input type="number" min="0" class="mbc-children" value="" />
                </div>

                <div class="mbc-field">
                    <label>Gemiddelde ticketprijs (per persoon per bezoek)</label>
                    <input type="text" inputmode="decimal" class="mbc-ticket" value="" placeholder="17,50" />
                </div>
                <div class="mbc-field">
                    <label>Bezoeken per maand (optioneel)</label>
                    <input type="number" min="0" step="1" class="mbc-per-month" value="0" />
                </div>
            </div>

            <button type="button" class="mbc-btn">BEREKEN BREAKEVEN</button>

            <div class="mbc-result" aria-live="polite"></div>
            <div class="mbc-note">Let op: dit is een indicatie. Sommige musea vragen toeslagen voor speciale exposities.</div>
        </div>
    </div>';
}
add_shortcode('museumkaart_breakeven', 'mbc_shortcode');
