<?php
/**
 * Plugin Name: Bank Vergelijker
 * Description: Bankvergelijker met scenario’s + “beste keuze” en beheerbare banken (links/logo’s) via WP Admin.
 * Version: 18.0
 * Author: Alexander van Dijl
 */

if (!defined('ABSPATH')) exit;

function bv_assets_base() { return plugin_dir_url(__FILE__) . 'assets/'; }

function bv_default_banks() {
  $a = bv_assets_base();
  return [
    // Challenger
    [
      'id' => 'revolut',
      'name' => 'Revolut',
      'group' => 'challenger',
      'description' => 'Snel & internationaal sterk. Handig voor reizen en valuta.',
      'cta_text' => 'Open Revolut',
      'cta_url' => 'https://link.vraagalex.com/revolut',
      'logo_url' => $a.'revolut.svg',
      'badges' => [
        'Snel & gemak',
        ['label' => 'Let op: Geldmaat kan een service fee rekenen bij cash opnames met niet-NL banken (vaak €4).', 'type' => 'warning'],
        'NL IBAN mogelijk',
      ],
      'scores' => ['speed'=>5,'cash'=>2,'travel'=>5,'traditional'=>2,'business'=>3],
    ],
    [
      'id' => 'bunq',
      'name' => 'bunq',
      'group' => 'challenger',
      'description' => 'Nederlandse app-bank, snel geopend, veel subrekeningen.',
      'cta_text' => 'Open bunq',
      'cta_url' => 'https://link.vraagalex.com/bunq',
      'logo_url' => $a.'bunq.svg',
      'badges' => ['Cash-vriendelijk','NL rekening','Veel subrekeningen'],
      'scores' => ['speed'=>4,'cash'=>5,'travel'=>4,'traditional'=>3,'business'=>4],
    ],
    [
      'id' => 'n26',
      'name' => 'N26',
      'group' => 'challenger',
      'description' => 'Strakke app-bank (EU) met goede basis en gebruiksgemak.',
      'cta_text' => 'Bekijk N26',
      'cta_url' => '',
      'logo_url' => $a.'n26.svg',
      'badges' => ['Simpel','EU bank'],
      'scores' => ['speed'=>4,'cash'=>2,'travel'=>4,'traditional'=>2,'business'=>2],
    ],

    // Traditional (SNS removed)
    [
      'id' => 'ing',
      'name' => 'ING',
      'group' => 'traditional',
      'description' => 'Grote Nederlandse bank, breed pakket en vertrouwd voor dagelijks gebruik.',
      'cta_text' => 'Bekijk ING',
      'cta_url' => '',
      'logo_url' => $a.'ing.svg',
      'badges' => ['Traditioneel NL','Breed aanbod'],
      'scores' => ['speed'=>3,'cash'=>4,'travel'=>3,'traditional'=>5,'business'=>4],
    ],
    [
      'id' => 'rabobank',
      'name' => 'Rabobank',
      'group' => 'traditional',
      'description' => 'Sterk in Nederland en zakelijk, breed dienstenpakket.',
      'cta_text' => 'Bekijk Rabobank',
      'cta_url' => '',
      'logo_url' => $a.'rabobank.svg',
      'badges' => ['Traditioneel NL','Zakelijk sterk'],
      'scores' => ['speed'=>3,'cash'=>4,'travel'=>3,'traditional'=>5,'business'=>5],
    ],
    [
      'id' => 'abn_amro',
      'name' => 'ABN AMRO',
      'group' => 'traditional',
      'description' => 'Brede bank voor particulier en zakelijk met veel ondersteuning.',
      'cta_text' => 'Bekijk ABN AMRO',
      'cta_url' => '',
      'logo_url' => $a.'abn_amro.svg',
      'badges' => ['Traditioneel NL'],
      'scores' => ['speed'=>3,'cash'=>4,'travel'=>3,'traditional'=>5,'business'=>5],
    ],
    [
      'id' => 'asn',
      'name' => 'ASN Bank',
      'group' => 'traditional',
      'description' => 'Bekend om duurzaamheid en Nederlandse basisbankzaken.',
      'cta_text' => 'Bekijk ASN',
      'cta_url' => '',
      'logo_url' => $a.'asn.svg',
      'badges' => ['Duurzaam','Traditioneel NL'],
      'scores' => ['speed'=>3,'cash'=>4,'travel'=>2,'traditional'=>4,'business'=>2],
    ],
    [
      'id' => 'knab',
      'name' => 'Knab',
      'group' => 'traditional',
      'description' => 'Populair bij ondernemers en freelancers, modern NL bankieren.',
      'cta_text' => 'Bekijk Knab',
      'cta_url' => '',
      'logo_url' => $a.'knab.svg',
      'badges' => ['Zakelijk','NL bank'],
      'scores' => ['speed'=>4,'cash'=>3,'travel'=>3,'traditional'=>4,'business'=>5],
    ],
  ];
}

function bv_migrate_banks_option() {
  $banks = get_option('bv_banks_data');
  if (!is_array($banks) || empty($banks)) {
    update_option('bv_banks_data', bv_default_banks());
    return;
  }

  $changed = false;
  $filtered = [];

  foreach ($banks as $b) {
    if (!is_array($b)) { $changed = true; continue; }
    $id = sanitize_title($b['id'] ?? '');
    if ($id === 'sns') { $changed = true; continue; } // remove SNS
    // Ensure Revolut link is present
    if ($id === 'revolut') {
      if (empty($b['cta_url']) || $b['cta_url'] !== 'https://link.vraagalex.com/revolut') {
        $b['cta_url'] = 'https://link.vraagalex.com/revolut';
        if (empty($b['cta_text'])) $b['cta_text'] = 'Open Revolut';
        $changed = true;
      }
    }
    $filtered[] = $b;
  }

  if ($changed) update_option('bv_banks_data', $filtered);
}
register_activation_hook(__FILE__, 'bv_migrate_banks_option');
add_action('admin_init', 'bv_migrate_banks_option');

function bv_get_banks() {
  $banks = get_option('bv_banks_data');
  if (!is_array($banks) || empty($banks)) {
    $banks = bv_default_banks();
    update_option('bv_banks_data', $banks);
  }
  return $banks;
}

function bv_register_assets() {
  $a = bv_assets_base();
  wp_register_style('bv-bankvergelijker', $a.'bv.css', [], '15.0');
  wp_register_script('bv-bankvergelijker', $a.'bv.js', [], '15.0', true);
  wp_register_script('bv-bankvergelijker-admin', $a.'admin.js', ['jquery'], '15.0', true);
}

add_action('wp_enqueue_scripts', 'bv_register_assets');
add_action('admin_enqueue_scripts', 'bv_register_assets');
add_action('admin_enqueue_scripts', function($hook){
  if ($hook !== 'settings_page_bank-vergelijker') return;
  wp_enqueue_media();
  wp_enqueue_script('bv-bankvergelijker-admin');
wp_enqueue_style('bv-bankvergelijker-admin', bv_assets_base().'admin.css', [], '18.0');
});

function bv_render_badge($badge) {
  $label = '';
  $type = 'normal';
  if (is_string($badge)) { $label = $badge; }
  elseif (is_array($badge)) {
    if (isset($badge['label']) && is_string($badge['label'])) $label = $badge['label'];
    if (isset($badge['type']) && is_string($badge['type'])) $type = $badge['type'];
  }
  if ($label === '') return '';
  $class = 'bv-badge'.($type==='warning' ? ' bv-badge-warning' : '');
  return '<span class="'.esc_attr($class).'">'.esc_html($label).'</span>';
}

function bv_render_logo($logo_url, $bank_name) {
  if (!$logo_url) return '';
  return '<div class="bv-bank-logo"><img loading="lazy" decoding="async" src="'.esc_url($logo_url).'" alt="'.esc_attr($bank_name.' logo').'" /></div>';
}

function bv_shortcode($atts = []) {
  wp_enqueue_style('bv-bankvergelijker');
  wp_enqueue_script('bv-bankvergelijker');

  $banks = bv_get_banks();
  $challengers = array_values(array_filter($banks, fn($b)=>($b['group']??'')==='challenger'));
  $traditionals = array_values(array_filter($banks, fn($b)=>($b['group']??'')==='traditional'));

  ob_start(); ?>
  <div class="bv-wrapper" data-bv>
    <h2 class="bv-title">Bankvergelijker</h2>
    <p class="bv-subtitle">Kies wat jij belangrijk vindt, dan rolt er meteen een “beste keuze” uit. Daarna kun je alle rekeningen rustig vergelijken.</p>

    <!-- OPTIES BOVEN -->
    <div class="bv-panel">
      <h3 class="bv-panel-title">Wat past bij jou?</h3>
      <div class="bv-chips">
        <button type="button" class="bv-chip" data-scenario="speed">Snel & gemak</button>
        <button type="button" class="bv-chip" data-scenario="cash">Cash in NL</button>
        <button type="button" class="bv-chip" data-scenario="travel">Reizen & valuta</button>
        <button type="button" class="bv-chip" data-scenario="traditional">Traditioneel NL</button>
        <button type="button" class="bv-chip" data-scenario="business">Zakelijk</button>
        <button type="button" class="bv-chip bv-chip-reset" data-reset="1">Reset</button>
      </div>

      <div class="bv-best">
        <h4 class="bv-best-title">Beste keuze (top 3)</h4>
        <p class="bv-best-note">Klik één of meer keuzes hierboven. Tip: klik meerdere chips aan. De tool telt je voorkeuren bij elkaar op.</p>
        <div class="bv-best-list" aria-live="polite"></div>
      </div>
    </div>

    <!-- RESULTATEN ERONDER -->
    <div class="bv-panel">
      <h3 class="bv-section-title">Challenger banken</h3>
      <div class="bv-cards">
        <?php foreach ($challengers as $b): ?>
          <div class="bv-card" data-bank="<?php echo esc_attr($b['id']); ?>" data-scores="<?php echo esc_attr(wp_json_encode($b['scores'] ?? [])); ?>">
            <?php echo bv_render_logo($b['logo_url'] ?? '', $b['name'] ?? ''); ?>
            <div class="bv-card-body">
              <h4 class="bv-card-title"><?php echo esc_html($b['name'] ?? ''); ?></h4>
              <p class="bv-card-desc"><?php echo esc_html($b['description'] ?? ''); ?></p>
              <div class="bv-badges">
                <?php foreach (($b['badges'] ?? []) as $badge) echo bv_render_badge($badge); ?>
              </div>
              <?php if (!empty($b['cta_url'])): ?>
                <a class="bv-cta" href="<?php echo esc_url($b['cta_url']); ?>" target="_blank" rel="nofollow noopener"><?php echo esc_html($b['cta_text'] ?? 'Open'); ?></a>
              <?php endif; ?>
            </div>
          </div>
        <?php endforeach; ?>
      </div>

      <h3 class="bv-section-title" style="margin-top:18px;">Traditionele banken</h3>
      <div class="bv-cards">
        <?php foreach ($traditionals as $b): ?>
          <div class="bv-card" data-bank="<?php echo esc_attr($b['id']); ?>" data-scores="<?php echo esc_attr(wp_json_encode($b['scores'] ?? [])); ?>">
            <?php echo bv_render_logo($b['logo_url'] ?? '', $b['name'] ?? ''); ?>
            <div class="bv-card-body">
              <h4 class="bv-card-title"><?php echo esc_html($b['name'] ?? ''); ?></h4>
              <p class="bv-card-desc"><?php echo esc_html($b['description'] ?? ''); ?></p>
              <div class="bv-badges">
                <?php foreach (($b['badges'] ?? []) as $badge) echo bv_render_badge($badge); ?>
              </div>
              <?php if (!empty($b['cta_url'])): ?>
                <a class="bv-cta" href="<?php echo esc_url($b['cta_url']); ?>" target="_blank" rel="nofollow noopener"><?php echo esc_html($b['cta_text'] ?? 'Bekijk'); ?></a>
              <?php endif; ?>
            </div>
          </div>
        <?php endforeach; ?>
      </div>

      <p class="bv-muted">Wil je links toevoegen of aanpassen? Ga naar <strong>Instellingen → Bank Vergelijker</strong>.</p>
      <p class="bv-disclaimer">Let op: prijzen en limieten kunnen wijzigen. Controleer altijd de actuele voorwaarden bij de aanbieder.</p>
    </div>
  </div>
  <?php
  return ob_get_clean();
}
add_shortcode('bank_vergelijker', 'bv_shortcode');

/** Admin page */
add_action('admin_menu', function(){
  add_options_page('Bank Vergelijker', 'Bank Vergelijker', 'manage_options', 'bank-vergelijker', 'bv_admin_page');
});

function bv_sanitize_banks($input) {
  if (!is_array($input)) return [];
  $out = [];
  foreach ($input as $row) {
    if (!is_array($row)) continue;
    $id = sanitize_title($row['id'] ?? '');
    $name = sanitize_text_field($row['name'] ?? '');
    if ($id === '' || $name === '' || $id === 'sns') continue;

    $group = (($row['group'] ?? 'challenger') === 'traditional') ? 'traditional' : 'challenger';
    $desc = sanitize_text_field($row['description'] ?? '');
    $cta_text = sanitize_text_field($row['cta_text'] ?? '');
    $cta_url = esc_url_raw($row['cta_url'] ?? '');
    $logo_url = esc_url_raw($row['logo_url'] ?? '');

    $badges = $row['badges'] ?? '';
    $badges_arr = [];
    if (is_string($badges)) {
      foreach (explode(',', $badges) as $b) { $b = trim($b); if ($b !== '') $badges_arr[] = $b; }
    }

    $scores_in = $row['scores'] ?? [];
    $scores = [];
    foreach (['speed','cash','travel','traditional','business'] as $k) {
      $v = 0;
      if (is_array($scores_in) && isset($scores_in[$k])) $v = intval($scores_in[$k]);
      $scores[$k] = max(0, min(5, $v));
    }

    if ($id === 'revolut') {
      $cta_url = 'https://link.vraagalex.com/revolut'; // force correct link
      if ($cta_text === '') $cta_text = 'Open Revolut';
    }

    $out[] = ['id'=>$id,'name'=>$name,'group'=>$group,'description'=>$desc,'cta_text'=>$cta_text,'cta_url'=>$cta_url,'logo_url'=>$logo_url,'badges'=>$badges_arr,'scores'=>$scores];
  }
  return $out;
}

function bv_admin_page() {
  if (!current_user_can('manage_options')) return;

  if (isset($_POST['bv_save']) && check_admin_referer('bv_save_banks')) {
    $banks = bv_sanitize_banks($_POST['banks'] ?? []);
    update_option('bv_banks_data', $banks);
    echo '<div class="updated"><p>Opgeslagen.</p></div>';
  }

  if (isset($_POST['bv_reset']) && check_admin_referer('bv_save_banks')) {
    update_option('bv_banks_data', bv_default_banks());
    echo '<div class="updated"><p>Teruggezet naar standaard.</p></div>';
  }

  $banks = bv_get_banks();
  ?>
  <div class="wrap">
    <h1>Bank Vergelijker</h1>
    <div class="bv-admin-card">
    <p class="bv-help">Hier kun je per bank de <strong>link</strong> en <strong>logo</strong> instellen. <strong>SNS wordt automatisch verwijderd</strong> (bestaat niet meer).</p>

    </div>

    <form method="post">
      <?php wp_nonce_field('bv_save_banks'); ?>

      <table class="widefat fixed" id="bv-banks-table" style="margin-top:12px;">
        <thead>
          <tr>
            <th style="width:110px;">ID</th>
            <th>Naam</th>
            <th style="width:120px;">Groep</th>
            <th>Logo</th>
            <th>Link (CTA)</th>
            <th>Badges (komma)</th>
            <th style="width:260px;">Scores (0-5)</th>
            <th class="bv-actions"></th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($banks as $i=>$b): ?>
            <tr>
              <td><input name="banks[<?php echo $i; ?>][id]" value="<?php echo esc_attr($b['id']); ?>" class="regular-text" /></td>
              <td>
                <input name="banks[<?php echo $i; ?>][name]" value="<?php echo esc_attr($b['name']); ?>" class="regular-text" />
                <br><small><input name="banks[<?php echo $i; ?>][description]" value="<?php echo esc_attr($b['description'] ?? ''); ?>" class="regular-text" placeholder="Korte beschrijving" /></small>
              </td>
              <td>
                <select name="banks[<?php echo $i; ?>][group]">
                  <option value="challenger" <?php selected(($b['group']??'')==='challenger'); ?>>Challenger</option>
                  <option value="traditional" <?php selected(($b['group']??'')==='traditional'); ?>>Traditioneel</option>
                </select>
              </td>
              <td>
                <input name="banks[<?php echo $i; ?>][logo_url]" value="<?php echo esc_attr($b['logo_url'] ?? ''); ?>" class="regular-text bv-logo-url" placeholder="Logo URL" />
                <button class="button bv-pick-logo">Kies logo</button>
              </td>
              <td>
                <input name="banks[<?php echo $i; ?>][cta_text]" value="<?php echo esc_attr($b['cta_text'] ?? ''); ?>" class="regular-text" placeholder="Knoptekst" />
                <br><small><input name="banks[<?php echo $i; ?>][cta_url]" value="<?php echo esc_attr($b['cta_url'] ?? ''); ?>" class="regular-text" placeholder="https://..." <?php echo ($b['id']==='revolut')?'readonly':''; ?> /></small>
                <?php if ($b['id']==='revolut'): ?><br><small>Revolut-link staat vast op jouw affiliate link.</small><?php endif; ?>
              </td>
              <td>
                <?php $badges = $b['badges'] ?? []; $badges_str = is_array($badges) ? implode(', ', array_map('strval',$badges)) : ''; ?>
                <input name="banks[<?php echo $i; ?>][badges]" value="<?php echo esc_attr($badges_str); ?>" class="regular-text" />
              </td>
              <td>
                <?php $s = $b['scores'] ?? []; ?>
                <div style="display:grid;grid-template-columns:repeat(5, 1fr);gap:6px;align-items:center;">
                  <input type="number" min="0" max="5" name="banks[<?php echo $i; ?>][scores][speed]" value="<?php echo intval($s['speed'] ?? 0); ?>" title="Snel" />
                  <input type="number" min="0" max="5" name="banks[<?php echo $i; ?>][scores][cash]" value="<?php echo intval($s['cash'] ?? 0); ?>" title="Cash" />
                  <input type="number" min="0" max="5" name="banks[<?php echo $i; ?>][scores][travel]" value="<?php echo intval($s['travel'] ?? 0); ?>" title="Reizen" />
                  <input type="number" min="0" max="5" name="banks[<?php echo $i; ?>][scores][traditional]" value="<?php echo intval($s['traditional'] ?? 0); ?>" title="Traditioneel" />
                  <input type="number" min="0" max="5" name="banks[<?php echo $i; ?>][scores][business]" value="<?php echo intval($s['business'] ?? 0); ?>" title="Zakelijk" />
                </div>
                <small>Snel / Cash / Reizen / Trad / Zakelijk</small>
              </td>
              <td><button type="button" class="button bv-remove-bank">Verwijder</button></td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>

      <p><button id="bv-add-bank" class="button">+ Bank toevoegen</button></p>

      <p>
        <button type="submit" name="bv_save" class="button button-primary">Opslaan</button>
        <button type="submit" name="bv_reset" class="button">Reset naar standaard</button>
      </p>
    </form>

    <script type="text/html" id="bv-row-template">
      <tr>
        <td><input name="banks[][id]" value="" class="regular-text" placeholder="bijv. wise" /></td>
        <td>
          <input name="banks[][name]" value="" class="regular-text" placeholder="Naam" />
          <br><small><input name="banks[][description]" value="" class="regular-text" placeholder="Korte beschrijving" /></small>
        </td>
        <td>
          <select name="banks[][group]">
            <option value="challenger">Challenger</option>
            <option value="traditional">Traditioneel</option>
          </select>
        </td>
        <td>
          <input name="banks[][logo_url]" value="" class="regular-text bv-logo-url" placeholder="Logo URL" />
          <button class="button bv-pick-logo">Kies logo</button>
        </td>
        <td>
          <input name="banks[][cta_text]" value="Bekijk" class="regular-text" />
          <br><small><input name="banks[][cta_url]" value="" class="regular-text" placeholder="https://..." /></small>
        </td>
        <td><input name="banks[][badges]" value="" class="regular-text" placeholder="badge1, badge2" /></td>
        <td>
          <div style="display:grid;grid-template-columns:repeat(5, 1fr);gap:6px;align-items:center;">
            <input type="number" min="0" max="5" name="banks[][scores][speed]" value="3" />
            <input type="number" min="0" max="5" name="banks[][scores][cash]" value="3" />
            <input type="number" min="0" max="5" name="banks[][scores][travel]" value="3" />
            <input type="number" min="0" max="5" name="banks[][scores][traditional]" value="3" />
            <input type="number" min="0" max="5" name="banks[][scores][business]" value="3" />
          </div>
          <small>Snel / Cash / Reizen / Trad / Zakelijk</small>
        </td>
        <td><button type="button" class="button bv-remove-bank">Verwijder</button></td>
      </tr>
    </script>
  </div>
  <?php
}
