
(function () {
  function el(tag, attrs, children) {
    const node = document.createElement(tag);
    if (attrs) {
      for (const [k, v] of Object.entries(attrs)) {
        if (k === 'class') node.className = v;
        else if (k === 'html') node.innerHTML = v;
        else if (k.startsWith('on') && typeof v === 'function') node.addEventListener(k.substring(2).toLowerCase(), v);
        else node.setAttribute(k, v);
      }
    }
    (children || []).forEach(c => node.appendChild(typeof c === 'string' ? document.createTextNode(c) : c));
    return node;
  }

  function wallLabel(w) {
    return ({top:'Boven', right:'Rechts', bottom:'Onder', left:'Links'})[w] || 'Boven';
  }

  function openingRow(roomId, type, idx) {
    const id = `${roomId}-${type}-${idx}-${Math.random().toString(16).slice(2,8)}`;
    const row = el('div', { class: 'biq-opening-row', 'data-opening-id': id, 'data-opening-type': type });

    const wallSel = el('select', { 'data-f': 'wall' }, [
      el('option', { value: 'top' }, ['Boven']),
      el('option', { value: 'right' }, ['Rechts']),
      el('option', { value: 'bottom' }, ['Onder']),
      el('option', { value: 'left' }, ['Links']),
    ]);

    const anchorSel = el('select', { 'data-f': 'anchor' }, [
      el('option', { value: 'left' }, ['Links']),
      el('option', { value: 'center' }, ['Midden']),
      el('option', { value: 'right' }, ['Rechts']),
    ]);

    const distIn = el('input', { type: 'number', min: '0', step: '1', placeholder: 'Afstand (mm)', 'data-f': 'distance_mm' });
    const widthIn = el('input', { type: 'number', min: '0', step: '1', placeholder: 'Breedte (mm)', 'data-f': 'width_mm' });

    const centerDirSel = el('select', { 'data-f': 'center_direction', title: 'Alleen relevant bij anker = Midden' }, [
      el('option', { value: 'positive' }, ['→ rechts/beneden']),
      el('option', { value: 'negative' }, ['← links/boven']),
    ]);

    const uncertain = el('label', { class: 'biq-small' }, [
      el('input', { type: 'checkbox', 'data-f': 'uncertain' }),
      ' Onzeker'
    ]);

    row.appendChild(el('div', null, [el('label', null, [wallLabel('top')])]));
    row.firstChild.replaceWith(el('label', null, ['Wand', wallSel]));

    row.appendChild(el('label', null, ['Anker', anchorSel]));
    row.appendChild(el('label', null, ['Afstand', distIn]));
    row.appendChild(el('label', null, ['Breedte', widthIn]));
    row.appendChild(el('label', null, ['Midden-richting', centerDirSel]));

    // For windows, add bottom/height inputs (we’ll attach after row in wrapper)
    const wrapper = el('div', { class: 'biq-open-wrap' }, [row]);

    if (type === 'window') {
      const bottomIn = el('input', { type: 'number', min: '0', step: '1', placeholder: 'Onderkant v.a. vloer (mm)', 'data-f': 'bottom_from_floor_mm' });
      const hIn = el('input', { type: 'number', min: '0', step: '1', placeholder: 'Raamhoogte (mm)', 'data-f': 'height_mm' });

      const row2 = el('div', { class: 'biq-opening-row', 'data-opening-id': id, 'data-opening-type': type });
      row2.appendChild(el('label', null, ['Onderkant', bottomIn]));
      row2.appendChild(el('label', null, ['Hoogte', hIn]));
      row2.appendChild(el('div'));
      row2.appendChild(el('div', null, [uncertain]));
      row2.appendChild(el('div'));
      wrapper.appendChild(row2);
    } else {
      // doors: place uncertain nicely
      const row2 = el('div', { class: 'biq-btn-row' }, [uncertain]);
      wrapper.appendChild(row2);
    }

    const delBtn = el('button', { type: 'button', class: 'button', onClick: () => wrapper.remove() }, ['Verwijder']);
    const delRow = el('div', { class: 'biq-btn-row' }, [delBtn]);
    wrapper.appendChild(delRow);

    // Simple “drag beta” placeholder: not implemented yet but hook point
    const beta = el('div', { class: 'biq-small' }, ['(Slepen beta komt in v0.2 — nu invoeren via mm.)']);
    wrapper.appendChild(beta);

    // Show/hide center direction based on anchor
    function toggleCenterDir() {
      const isCenter = anchorSel.value === 'center';
      centerDirSel.disabled = !isCenter;
      centerDirSel.style.opacity = isCenter ? '1' : '0.5';
    }
    anchorSel.addEventListener('change', toggleCenterDir);
    toggleCenterDir();

    return wrapper;
  }

  function roomCard(type, idx) {
    const roomId = `${type}-${idx}`;

    const card = el('fieldset', { class: 'biq-card', 'data-room-id': roomId, 'data-room-type': type, 'data-room-index': String(idx) });

    const title = el('div', { class: 'biq-room-title' }, [
      el('h3', null, [ (type === 'bathroom' ? 'Badkamer ' : 'Toilet ') + idx ]),
      el('span', { class: 'biq-small' }, ['Afmetingen in mm'])
    ]);

    const grid = el('div', { class: 'biq-grid2' });

    const lIn = el('input', { type: 'number', min: '1', step: '1', required: 'required', 'data-f': 'length_mm' });
    const wIn = el('input', { type: 'number', min: '1', step: '1', required: 'required', 'data-f': 'width_mm' });
    const hIn = el('input', { type: 'number', min: '0', step: '1', 'data-f': 'height_mm', placeholder: 'Optioneel' });

    const notes = el('textarea', { rows: '3', 'data-f': 'notes', placeholder: 'Opmerkingen (optioneel)' });

    grid.appendChild(el('label', null, ['Lengte*', lIn]));
    grid.appendChild(el('label', null, ['Breedte*', wIn]));
    grid.appendChild(el('label', null, ['Hoogte', hIn]));
    grid.appendChild(el('label', null, ['Opmerking', notes]));

    const openings = el('div', { class: 'biq-openings' }, [
      el('h4', null, ['Deuren & ramen']),
      el('div', { class: 'biq-btn-row' }, [
        el('button', { type: 'button', class: 'button', onClick: () => openings.appendChild(openingRow(roomId, 'door', Date.now())) }, ['+ Deur toevoegen']),
        el('button', { type: 'button', class: 'button', onClick: () => openings.appendChild(openingRow(roomId, 'window', Date.now())) }, ['+ Raam toevoegen']),
      ]),
    ]);

    card.appendChild(title);
    card.appendChild(grid);
    card.appendChild(openings);

    return card;
  }

  function buildRooms() {
    const form = document.getElementById('biq-form');
    const roomsEl = document.getElementById('biq-rooms');
    roomsEl.innerHTML = '';

    const bc = Math.max(0, Math.min(5, parseInt(form.bathroom_count.value || '0', 10)));
    const tc = Math.max(0, Math.min(5, parseInt(form.toilet_count.value || '0', 10)));

    for (let i = 1; i <= bc; i++) roomsEl.appendChild(roomCard('bathroom', i));
    for (let i = 1; i <= tc; i++) roomsEl.appendChild(roomCard('toilet', i));

    if (bc + tc === 0) {
      roomsEl.appendChild(el('p', { class: 'biq-muted' }, ['Geen ruimtes gekozen.']));
    }
  }

  function readRooms() {
    const rooms = [];
    document.querySelectorAll('[data-room-id]').forEach(card => {
      const type = card.getAttribute('data-room-type');
      const index = parseInt(card.getAttribute('data-room-index'), 10);

      const length_mm = parseInt(card.querySelector('[data-f="length_mm"]').value || '0', 10);
      const width_mm  = parseInt(card.querySelector('[data-f="width_mm"]').value || '0', 10);
      const hRaw = card.querySelector('[data-f="height_mm"]').value;
      const height_mm = hRaw === '' ? '' : parseInt(hRaw || '0', 10);
      const notes = card.querySelector('[data-f="notes"]').value || '';

      const openings = [];
      card.querySelectorAll('.biq-open-wrap').forEach(wrap => {
        const row = wrap.querySelector('.biq-opening-row');
        if (!row) return;
        const otype = row.getAttribute('data-opening-type');

        const wall = row.querySelector('[data-f="wall"]').value;
        const anchor = row.querySelector('[data-f="anchor"]').value;
        const distance_mm = parseInt(row.querySelector('[data-f="distance_mm"]').value || '0', 10);
        const width_mm = row.querySelector('[data-f="width_mm"]').value;
        const center_direction = row.querySelector('[data-f="center_direction"]').value || 'positive';

        let uncertain = false;
        const uncBox = wrap.querySelector('[data-f="uncertain"]');
        if (uncBox) uncertain = !!uncBox.checked;

        const opening = {
          type: otype,
          wall,
          anchor,
          distance_mm,
          center_direction,
          width_mm: width_mm === '' ? '' : parseInt(width_mm || '0', 10),
          uncertain
        };

        if (otype === 'window') {
          const bottom = wrap.querySelector('[data-f="bottom_from_floor_mm"]').value;
          const oh = wrap.querySelector('[data-f="height_mm"]').value;
          opening.bottom_from_floor_mm = bottom === '' ? '' : parseInt(bottom || '0', 10);
          opening.height_mm = oh === '' ? '' : parseInt(oh || '0', 10);
        }

        openings.push(opening);
      });

      rooms.push({
        type,
        index,
        length_mm,
        width_mm,
        height_mm,
        notes,
        openings
      });
    });
    return rooms;
  }

  function setStatus(msg, kind) {
    const s = document.getElementById('biq-status');
    s.textContent = msg || '';
    s.classList.remove('biq-error', 'biq-success');
    if (kind === 'error') s.classList.add('biq-error');
    if (kind === 'success') s.classList.add('biq-success');
  }

  async function submitForm(ev) {
    ev.preventDefault();
    setStatus('Versturen…');

    const form = document.getElementById('biq-form');
    const payload = {
      qr_ref: (form.qr_ref.value || '').trim(),
      customer: {
        name: (form.customer_name.value || '').trim(),
        phone: (form.customer_phone.value || '').trim(),
        email: (form.customer_email.value || '').trim(),
        street: (form.customer_street.value || '').trim(),
        house_number: (form.customer_house_number.value || '').trim(),
        addition: (form.customer_addition.value || '').trim(),
        postcode: (form.customer_postcode.value || '').trim(),
        city: (form.customer_city.value || '').trim(),
      },
      counts: {
        bathrooms: parseInt(form.bathroom_count.value || '0', 10),
        toilets: parseInt(form.toilet_count.value || '0', 10),
      },
      general_notes: (form.general_notes.value || '').trim(),
      rooms: readRooms()
    };

    // Basic validation
    if (!payload.customer.name || !payload.customer.phone || !payload.customer.email) {
      setStatus('Vul klantgegevens volledig in.', 'error');
      return;
    }
    if (payload.rooms.length < 1) {
      setStatus('Klik “Maak ruimtes” en vul minimaal 1 ruimte in.', 'error');
      return;
    }
    for (const r of payload.rooms) {
      if (!r.length_mm || !r.width_mm) {
        setStatus('Vul bij elke ruimte lengte en breedte in (mm).', 'error');
        return;
      }
    }

    try {
      const res = await fetch(BIQ.restUrl, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-WP-Nonce': BIQ.nonce || ''
        },
        body: JSON.stringify(payload)
      });
      const out = await res.json();
      if (!res.ok || !out.ok) {
        setStatus(out.error || 'Versturen mislukt.', 'error');
        return;
      }
      setStatus(`Verstuurd! Intake-code: ${out.intake_code}`, 'success');
      form.reset();
      document.getElementById('biq-rooms').innerHTML = '';
    } catch (e) {
      setStatus('Netwerkfout bij versturen.', 'error');
    }
  }

  document.addEventListener('DOMContentLoaded', function () {
    const buildBtn = document.getElementById('biq-build');
    if (buildBtn) buildBtn.addEventListener('click', buildRooms);

    const form = document.getElementById('biq-form');
    if (form) form.addEventListener('submit', submitForm);
  });
})();
