<?php
/**
 * Plugin Name: Bathroom Intake (QR)
 * Description: Simple bathroom/toilet intake form with dimensions + doors/windows. Sends intake by email. Shortcode: [bathroom_intake]
 * Version: 0.1.0
 * Author: Alexander van Dijl (MVP)
 */

if (!defined('ABSPATH')) exit;

class Bathroom_Intake_QR {
  const OPT_TO_EMAIL = 'biq_to_email';
  const OPT_SUBJECT_PREFIX = 'biq_subject_prefix';
  const REST_NS = 'bathroom-intake/v1';

  public static function init() {
    add_action('admin_menu', [__CLASS__, 'admin_menu']);
    add_action('admin_init', [__CLASS__, 'register_settings']);
    add_shortcode('bathroom_intake', [__CLASS__, 'render_shortcode']);
    add_action('wp_enqueue_scripts', [__CLASS__, 'enqueue_assets']);
    add_action('rest_api_init', [__CLASS__, 'register_routes']);
  }

  public static function defaults() {
    return [
      self::OPT_TO_EMAIL => 'a.dijl@plieger.nl',
      self::OPT_SUBJECT_PREFIX => 'Intake'
    ];
  }

  public static function get_opt($key) {
    $defaults = self::defaults();
    $val = get_option($key, $defaults[$key] ?? '');
    if (empty($val) && isset($defaults[$key])) return $defaults[$key];
    return $val;
  }

  public static function register_settings() {
    register_setting('biq_settings_group', self::OPT_TO_EMAIL, [
      'type' => 'string',
      'sanitize_callback' => function($v) { return sanitize_email($v); },
      'default' => self::defaults()[self::OPT_TO_EMAIL],
    ]);
    register_setting('biq_settings_group', self::OPT_SUBJECT_PREFIX, [
      'type' => 'string',
      'sanitize_callback' => function($v) { return sanitize_text_field($v); },
      'default' => self::defaults()[self::OPT_SUBJECT_PREFIX],
    ]);
  }

  public static function admin_menu() {
    add_options_page(
      'Bathroom Intake',
      'Bathroom Intake',
      'manage_options',
      'bathroom-intake',
      [__CLASS__, 'settings_page']
    );
  }

  public static function settings_page() {
    if (!current_user_can('manage_options')) return;
    ?>
    <div class="wrap">
      <h1>Bathroom Intake instellingen</h1>
      <form method="post" action="options.php">
        <?php settings_fields('biq_settings_group'); ?>
        <table class="form-table" role="presentation">
          <tr>
            <th scope="row"><label for="<?php echo esc_attr(self::OPT_TO_EMAIL); ?>">Ontvangst e-mailadres</label></th>
            <td>
              <input type="email" class="regular-text" id="<?php echo esc_attr(self::OPT_TO_EMAIL); ?>" name="<?php echo esc_attr(self::OPT_TO_EMAIL); ?>" value="<?php echo esc_attr(self::get_opt(self::OPT_TO_EMAIL)); ?>" />
              <p class="description">Alle intakes worden hierheen gestuurd.</p>
            </td>
          </tr>
          <tr>
            <th scope="row"><label for="<?php echo esc_attr(self::OPT_SUBJECT_PREFIX); ?>">Onderwerp prefix</label></th>
            <td>
              <input type="text" class="regular-text" id="<?php echo esc_attr(self::OPT_SUBJECT_PREFIX); ?>" name="<?php echo esc_attr(self::OPT_SUBJECT_PREFIX); ?>" value="<?php echo esc_attr(self::get_opt(self::OPT_SUBJECT_PREFIX)); ?>" />
              <p class="description">Voorbeeld: “Intake”.</p>
            </td>
          </tr>
        </table>
        <?php submit_button(); ?>
      </form>

      <hr/>
      <h2>Shortcode</h2>
      <p>Plaats deze shortcode op een pagina:</p>
      <code>[bathroom_intake]</code>
      <p>Tip: maak een QR naar <code>/intake/?qr=AMST-OPST-12</code> (qr is optioneel).</p>
    </div>
    <?php
  }

  public static function enqueue_assets() {
    // Only load on pages that contain the shortcode.
    if (!is_singular()) return;
    global $post;
    if (!$post || stripos($post->post_content, '[bathroom_intake') === false) return;

    $ver = '0.1.0';
    wp_enqueue_style('biq_intake_css', plugin_dir_url(__FILE__) . 'assets/intake.css', [], $ver);
    wp_enqueue_script('biq_intake_js', plugin_dir_url(__FILE__) . 'assets/intake.js', [], $ver, true);

    wp_localize_script('biq_intake_js', 'BIQ', [
      'restUrl' => esc_url_raw(rest_url(self::REST_NS . '/submit')),
      'nonce' => wp_create_nonce('wp_rest'),
    ]);
  }

  public static function render_shortcode($atts = []) {
    $qr = isset($_GET['qr']) ? sanitize_text_field($_GET['qr']) : '';
    ob_start();
    ?>
    <div class="biq">
      <h2>Badkamer / Toilet intake</h2>
      <p class="biq-muted">Vul je gegevens in. Afmetingen in <strong>mm</strong>. Hoogte is optioneel.</p>

      <form id="biq-form">
        <input type="hidden" name="qr_ref" value="<?php echo esc_attr($qr); ?>"/>

        <fieldset class="biq-card">
          <legend>Klantgegevens</legend>
          <div class="biq-grid2">
            <label>Naam*<input required name="customer_name" type="text" /></label>
            <label>Telefoon*<input required name="customer_phone" type="tel" /></label>
            <label>E-mail*<input required name="customer_email" type="email" /></label>
            <span></span>

            <label>Straat*<input required name="customer_street" type="text" /></label>
            <label>Huisnr*<input required name="customer_house_number" type="text" /></label>
            <label>Toevoeging<input name="customer_addition" type="text" /></label>
            <label>Postcode*<input required name="customer_postcode" type="text" placeholder="1234AB" /></label>
            <label>Plaats*<input required name="customer_city" type="text" /></label>
            <span></span>
          </div>
        </fieldset>

        <fieldset class="biq-card">
          <legend>Aantallen</legend>
          <div class="biq-grid2">
            <label>Badkamers<input name="bathroom_count" type="number" min="0" max="5" value="1"/></label>
            <label>Toiletten<input name="toilet_count" type="number" min="0" max="5" value="0"/></label>
          </div>
          <button type="button" class="button" id="biq-build">Maak ruimtes</button>
          <p class="biq-muted">Klik “Maak ruimtes” nadat je aantallen hebt ingevuld.</p>
        </fieldset>

        <div id="biq-rooms"></div>

        <fieldset class="biq-card">
          <legend>Verzenden</legend>
          <label>Algemene opmerkingen (optioneel)
            <textarea name="general_notes" rows="4" placeholder="Bijv. voorkeuren, planning, budget, etc."></textarea>
          </label>
          <div class="biq-actions">
            <button type="submit" class="button button-primary">Stuur naar adviseur</button>
            <span id="biq-status" class="biq-muted"></span>
          </div>
        </fieldset>
      </form>

      <details class="biq-muted">
        <summary>Wat gebeurt er met mijn gegevens?</summary>
        <p>Je intake wordt per e-mail verstuurd naar een adviseur om een offerte voor te bereiden.</p>
      </details>
    </div>
    <?php
    return ob_get_clean();
  }

  public static function register_routes() {
    register_rest_route(self::REST_NS, '/submit', [
      'methods'  => 'POST',
      'permission_callback' => '__return_true', // public form
      'callback' => [__CLASS__, 'handle_submit'],
      'args' => [],
    ]);
  }

  private static function intake_code() {
    $dt = new DateTime('now', wp_timezone());
    $stamp = $dt->format('Ymd-Hi');
    $rand = strtoupper(substr(bin2hex(random_bytes(3)), 0, 6)); // 6 hex chars
    return "PLG-INTAKE-{$stamp}-{$rand}";
  }

  private static function safe_int($v) {
    if ($v === null || $v === '') return null;
    return intval($v);
  }

  private static function normalize_wall($wall) {
    $wall = strtolower((string)$wall);
    $allowed = ['top','right','bottom','left'];
    return in_array($wall, $allowed, true) ? $wall : 'top';
  }

  private static function normalize_anchor($a) {
    $a = strtolower((string)$a);
    $allowed = ['left','center','right'];
    return in_array($a, $allowed, true) ? $a : 'left';
  }

  private static function anchor_to_offset_from_left($anchor, $distance_mm, $wall_length_mm, $center_dir) {
    // Always return absolute offset from "left/begin" of the wall.
    $anchor = self::normalize_anchor($anchor);
    $d = max(0, intval($distance_mm));
    $L = max(0, intval($wall_length_mm));

    if ($L <= 0) return $d;

    if ($anchor === 'left') return $d;
    if ($anchor === 'right') return max(0, $L - $d);

    // center
    $mid = $L / 2.0;
    $dir = ($center_dir === 'negative') ? -1 : 1;
    $offset = $mid + ($dir * $d);
    if ($offset < 0) $offset = 0;
    if ($offset > $L) $offset = $L;
    return intval(round($offset));
  }

  public static function handle_submit(WP_REST_Request $req) {
    $nonce = $req->get_header('X-WP-Nonce');
    // Nonce is best-effort; allow if missing to keep QR form functional without auth.
    if ($nonce && !wp_verify_nonce($nonce, 'wp_rest')) {
      return new WP_REST_Response(['ok' => false, 'error' => 'Invalid nonce'], 403);
    }

    $data = $req->get_json_params();
    if (!is_array($data)) {
      return new WP_REST_Response(['ok' => false, 'error' => 'Invalid JSON'], 400);
    }

    // Extract
    $qr_ref = isset($data['qr_ref']) ? sanitize_text_field($data['qr_ref']) : '';
    $customer = $data['customer'] ?? [];
    $counts = $data['counts'] ?? [];
    $rooms = $data['rooms'] ?? [];
    $general_notes = isset($data['general_notes']) ? sanitize_textarea_field($data['general_notes']) : '';

    // Basic validation
    $name = sanitize_text_field($customer['name'] ?? '');
    $phone = sanitize_text_field($customer['phone'] ?? '');
    $email = sanitize_email($customer['email'] ?? '');
    $street = sanitize_text_field($customer['street'] ?? '');
    $house_number = sanitize_text_field($customer['house_number'] ?? '');
    $addition = sanitize_text_field($customer['addition'] ?? '');
    $postcode = sanitize_text_field($customer['postcode'] ?? '');
    $city = sanitize_text_field($customer['city'] ?? '');

    if (!$name || !$phone || !$email || !$street || !$house_number || !$postcode || !$city) {
      return new WP_REST_Response(['ok' => false, 'error' => 'Missing required customer fields'], 400);
    }

    $bathrooms = intval($counts['bathrooms'] ?? 0);
    $toilets = intval($counts['toilets'] ?? 0);

    if (!is_array($rooms) || count($rooms) < 1) {
      return new WP_REST_Response(['ok' => false, 'error' => 'No rooms provided'], 400);
    }

    $code = self::intake_code();
    $dt = new DateTime('now', wp_timezone());
    $timestamp = $dt->format('Y-m-d H:i');

    // Build email
    $subject_prefix = self::get_opt(self::OPT_SUBJECT_PREFIX);
    $subject = trim($subject_prefix) . " – {$bathrooms} badk / {$toilets} toilet – {$name} – {$code}";

    $lines = [];
    $lines[] = "Intake-code: {$code}";
    $lines[] = "Timestamp (Europe/Amsterdam): {$timestamp}";
    $lines[] = "QR-ref: " . ($qr_ref ? $qr_ref : '-');
    $lines[] = "";
    $lines[] = "Klantgegevens";
    $lines[] = "Naam: {$name}";
    $lines[] = "Telefoon: {$phone}";
    $lines[] = "E-mail: {$email}";
    $addr = trim($street . ' ' . $house_number . ($addition ? ' ' . $addition : ''));
    $lines[] = "Adres: {$addr}";
    $lines[] = "Postcode/Plaats: {$postcode} {$city}";
    $lines[] = "";
    $lines[] = "Aantallen";
    $lines[] = "Badkamers: {$bathrooms}";
    $lines[] = "Toiletten: {$toilets}";
    $lines[] = "";

    $export = [];
    $export[] = "TYPE|INTAKE_CODE|QR_REF|ROOM_TYPE|ROOM_INDEX|L_MM|W_MM|H_MM|NOTES|OPENING_TYPE|WALL|ANCHOR|DIST_MM|CENTER_DIR|OFFSET_FROM_LEFT_MM|WIDTH_MM|BOTTOM_FROM_FLOOR_MM|OPENING_H_MM|UNCERTAIN";

    foreach ($rooms as $room) {
      $rtype = sanitize_text_field($room['type'] ?? 'bathroom');
      $rindex = intval($room['index'] ?? 1);
      $L = intval($room['length_mm'] ?? 0);
      $W = intval($room['width_mm'] ?? 0);
      $H = isset($room['height_mm']) && $room['height_mm'] !== '' ? intval($room['height_mm']) : null;
      $rnotes = sanitize_textarea_field($room['notes'] ?? '');

      $lines[] = strtoupper($rtype) . " {$rindex}";
      $dims = "{$L} x {$W} mm" . ($H ? " x {$H} mm" : "");
      $lines[] = "Afmetingen: {$dims}";
      if ($rnotes) $lines[] = "Opmerking: {$rnotes}";

      $openings = $room['openings'] ?? [];
      if (!is_array($openings) || count($openings) === 0) {
        $lines[] = "Openingen: -";
        $lines[] = "";
        $export[] = "ROOM|{$code}|{$qr_ref}|{$rtype}|{$rindex}|{$L}|{$W}|" . ($H ?? '') . "|" . str_replace(["\n","\r","|"], [' ',' ','/'], $rnotes) . "||||||||||||";
        continue;
      }

      // Group output
      $doors = [];
      $windows = [];
      foreach ($openings as $op) {
        $otype = sanitize_text_field($op['type'] ?? '');
        if ($otype === 'window') $windows[] = $op; else $doors[] = $op;
      }

      if (count($doors) > 0) {
        $lines[] = "Deuren:";
        foreach ($doors as $d) {
          $wall = self::normalize_wall($d['wall'] ?? 'top');
          $anchor = self::normalize_anchor($d['anchor'] ?? 'left');
          $dist = intval($d['distance_mm'] ?? 0);
          $cdir = ($d['center_direction'] ?? 'positive') === 'negative' ? 'negative' : 'positive';
          $width = isset($d['width_mm']) && $d['width_mm'] !== '' ? intval($d['width_mm']) : null;
          $unc = !empty($d['uncertain']) ? 'ja' : 'nee';

          $wall_len = ($wall === 'top' || $wall === 'bottom') ? $L : $W;
          $offset = self::anchor_to_offset_from_left($anchor, $dist, $wall_len, $cdir);

          $lines[] = "- Wand: {$wall}, Positie: {$anchor} {$dist}mm" . ($anchor === 'center' ? " ({$cdir})" : "") . ", Breedte: " . ($width ? "{$width}mm" : "-") . ", Onzeker: {$unc}";
          $export[] = "OPENING|{$code}|{$qr_ref}|{$rtype}|{$rindex}|{$L}|{$W}|" . ($H ?? '') . "|" . str_replace(["\n","\r","|"], [' ',' ','/'], $rnotes)
            . "|door|{$wall}|{$anchor}|{$dist}|{$cdir}|{$offset}|".($width ?? '')."|||{$unc}";
        }
      }

      if (count($windows) > 0) {
        $lines[] = "Ramen:";
        foreach ($windows as $w) {
          $wall = self::normalize_wall($w['wall'] ?? 'top');
          $anchor = self::normalize_anchor($w['anchor'] ?? 'left');
          $dist = intval($w['distance_mm'] ?? 0);
          $cdir = ($w['center_direction'] ?? 'positive') === 'negative' ? 'negative' : 'positive';
          $width = isset($w['width_mm']) && $w['width_mm'] !== '' ? intval($w['width_mm']) : null;
          $bottom = isset($w['bottom_from_floor_mm']) && $w['bottom_from_floor_mm'] !== '' ? intval($w['bottom_from_floor_mm']) : null;
          $wh = isset($w['height_mm']) && $w['height_mm'] !== '' ? intval($w['height_mm']) : null;
          $unc = !empty($w['uncertain']) ? 'ja' : 'nee';

          $wall_len = ($wall === 'top' || $wall === 'bottom') ? $L : $W;
          $offset = self::anchor_to_offset_from_left($anchor, $dist, $wall_len, $cdir);

          $lines[] = "- Wand: {$wall}, Positie: {$anchor} {$dist}mm" . ($anchor === 'center' ? " ({$cdir})" : "") . ", Breedte: " . ($width ? "{$width}mm" : "-") . ", Onderkant: " . ($bottom ? "{$bottom}mm" : "-") . ", Hoogte: " . ($wh ? "{$wh}mm" : "-") . ", Onzeker: {$unc}";

          $export[] = "OPENING|{$code}|{$qr_ref}|{$rtype}|{$rindex}|{$L}|{$W}|" . ($H ?? '') . "|" . str_replace(["\n","\r","|"], [' ',' ','/'], $rnotes)
            . "|window|{$wall}|{$anchor}|{$dist}|{$cdir}|{$offset}|".($width ?? '')."|".($bottom ?? '')."|".($wh ?? '')."|{$unc}";
        }
      }

      $lines[] = "";
    }

    if ($general_notes) {
      $lines[] = "Algemene opmerkingen";
      $lines[] = $general_notes;
      $lines[] = "";
    }

    $lines[] = "----- | EXPORT (voor later importeren) -----";
    foreach ($export as $row) $lines[] = $row;

    $body = implode("\n", $lines);

    $to = self::get_opt(self::OPT_TO_EMAIL);
    if (!$to) $to = self::defaults()[self::OPT_TO_EMAIL];

    $headers = [
      'Content-Type: text/plain; charset=UTF-8',
      'Reply-To: ' . $name . ' <' . $email . '>',
    ];

    $sent = wp_mail($to, $subject, $body, $headers);

    if (!$sent) {
      return new WP_REST_Response(['ok' => false, 'error' => 'Mail sending failed'], 500);
    }

    return new WP_REST_Response(['ok' => true, 'intake_code' => $code, 'timestamp' => $timestamp], 200);
  }
}

Bathroom_Intake_QR::init();
