(function(){
  function $(sel, ctx){ return (ctx||document).querySelector(sel); }
  function $all(sel, ctx){ return Array.prototype.slice.call((ctx||document).querySelectorAll(sel)); }

  function normalizeForUI(raw){
    return (raw||'').trim();
  }

  function buildDial(raw){
    // Mirror PHP rules for display:
    raw = (raw||'').trim();
    if(!raw) return '—';
    let cleaned = raw.replace(/[()\s\.\-–—]+/g,'');
    if(cleaned.startsWith('+')){
      return '00' + cleaned.substring(1).replace(/\D+/g,'').replace(/^0+/,'');
    }
    if(cleaned.startsWith('00')){
      return '00' + cleaned.substring(2).replace(/\D+/g,'');
    }
    return cleaned.replace(/\D+/g,'');
  }

  function makeHref(access, target){
    if(!access) return '#';
    if(target) return 'tel:' + access + ',,' + target;
    return 'tel:' + access;
  }

  async function post(url, data){
    const fd = new FormData();
    Object.keys(data).forEach(k=>fd.append(k, data[k]));
    const res = await fetch(url, {method:'POST', credentials:'same-origin', body: fd});
    return res.json();
  }

  function show(el){ if(el) el.hidden = false; }
  function hide(el){ if(el) el.hidden = true; }

  function setMsg(card, msg){
    const box = $('[data-kcs-msg]', card);
    if(box) box.textContent = msg || '';
  }

  async function doCheck(card){
    const input = $('.kcs-input', card);
    const raw = input ? input.value : '';
    if(!raw || !raw.trim()){ setMsg(card, KCS.msg_empty); return; }

    setMsg(card, '');
    const payload = {
      action: 'kcs_check',
      nonce: KCS.nonce,
      number: raw,
      page_url: location.href,
      post_id: (document.body.dataset.postId || '')
    };

    let json;
    try{
      json = await post(KCS.ajax_url, payload);
    } catch(e){
      setMsg(card, 'Oeps—checken lukte niet. Probeer later opnieuw.');
      return;
    }

    if(!json || !json.success){
      setMsg(card, KCS.msg_invalid);
      return;
    }

    const data = json.data;
    show($('[data-kcs-result]', card));
    $('[data-kcs-number]', card).textContent = normalizeForUI(raw);
    $('[data-kcs-type]', card).textContent = data.type || '—';
    $('[data-kcs-note]', card).textContent = data.note || '—';

    // CTA always shown after check
    const cta = $('[data-kcs-cta]', card);
    show(cta);

    // build dtmf target for call buttons
    const dtmf = (data.extra && data.extra.dtmf) ? data.extra.dtmf : buildDial(raw);

    const btnNL = $('[data-kcs-call="nl"]', card);
    const btnUK = $('[data-kcs-call="uk"]', card);
    const btnUS = $('[data-kcs-call="us"]', card);

    if(btnNL) btnNL.setAttribute('href', makeHref(KCS.outcall, dtmf));
    if(btnUK) btnUK.setAttribute('href', makeHref(KCS.access_uk, dtmf));
    if(btnUS) btnUS.setAttribute('href', makeHref(KCS.access_us, dtmf));
  }

  function openHelp(card){
    const help = $('[data-kcs-help]', card);
    const input = $('.kcs-input', card);
    const raw = input ? input.value : '';
    const dial = buildDial(raw);
    const dialEl = $('[data-kcs-dial]', card);
    if(dialEl) dialEl.textContent = dial;
    show(help);

    // log
    post(KCS.ajax_url, {
      action:'kcs_log', nonce:KCS.nonce, event:'help_open',
      number: raw, page_url:location.href
    }).catch(()=>{});
  }
  function closeHelp(card){ hide($('[data-kcs-help]', card)); }

  function logCall(event, raw, access, target, rawFound){
    return post(KCS.ajax_url, {
      action:'kcs_log', nonce:KCS.nonce, event:event,
      number: raw || '', access: access||'', target: target||'', raw_found: rawFound||'',
      page_url: location.href
    }).catch(()=>{});
  }

  function initCard(card){
    card.addEventListener('click', (e)=>{
      const btn = e.target.closest('[data-kcs-action]');
      if(btn){
        const action = btn.getAttribute('data-kcs-action');
        if(action === 'check') doCheck(card);
        if(action === 'call'){
          // just log, actual call is handled by CTA buttons after check
          const raw = $('.kcs-input', card).value || '';
          logCall('call_click_main', raw, '', buildDial(raw), '');
          setMsg(card, 'Tip: klik op één van de NL/UK/US knoppen onder het resultaat.');
        }
        if(action === 'help') openHelp(card);
      }
      if(e.target.closest('[data-kcs-close]')) closeHelp(card);
    });
  }

  function initCallouts(){
    $all('[data-kcs-callout]').forEach(box=>{
      const rawFound = box.getAttribute('data-kcs-raw') || '';
      const target = box.getAttribute('data-kcs-target') || '';
      const helpBtn = box.querySelector('[data-kcs-callout-help]');
      const helpBox = box.querySelector('[data-kcs-help-inline]');
      const closeBtn = box.querySelector('[data-kcs-callout-close]');
      const dialEl = box.querySelector('[data-kcs-callout-dial]');

      if(helpBtn && helpBox){
        helpBtn.addEventListener('click', ()=>{
          if(dialEl) dialEl.textContent = target || '—';
          helpBox.hidden = false;
          logCall('callout_help_open', '', '', target, rawFound);
        });
      }
      if(closeBtn && helpBox){
        closeBtn.addEventListener('click', ()=> helpBox.hidden = true);
      }

      // log clicks on callout buttons
      $all('[data-kcs-callout-call]', box).forEach(a=>{
        a.addEventListener('click', ()=>{
          const access = a.getAttribute('data-kcs-callout-call') || '';
          logCall('callout_call_click', '', access, target, rawFound);
        });
      });
    });
  }

  document.addEventListener('DOMContentLoaded', ()=>{
    $all('[data-kcs]').forEach(initCard);
    initCallouts();
  });
})();