<?php
if (!defined('ABSPATH')) exit;

/**
 * Normalize a number-like string:
 * - keep leading '+' or '00' if present
 * - strip spaces, dots, dashes, parentheses
 */
function kcs_normalize($raw) {
    $raw = trim((string)$raw);
    if ($raw === '') return '';
    $raw = preg_replace('/[()\s\.\-–—]+/u', '', $raw);

    // preserve + or 00 prefix
    if (strpos($raw, '+') === 0) {
        return '+' . preg_replace('/\D+/', '', substr($raw, 1));
    }
    if (strpos($raw, '00') === 0) {
        return '00' . preg_replace('/\D+/', '', substr($raw, 2));
    }
    return preg_replace('/\D+/', '', $raw);
}

/**
 * Determine what the user should dial after connecting (DTMF).
 * Rules:
 * - NL numbers: dial as NL (no country code)
 * - Foreign numbers: 00 + country code + rest
 * - If user types +.. -> convert to 00..
 */
function kcs_to_dtmf_target($raw) {
    $n = kcs_normalize($raw);
    if ($n === '') return '';
    if (strpos($n, '+') === 0) {
        return '00' . ltrim(substr($n, 1), '0');
    }
    if (strpos($n, '00') === 0) {
        return $n;
    }
    // assume national
    return $n;
}

function kcs_is_nl_mobile_or_fixed_from_international($norm) {
    // detect 0031... or +31... as NL and convert to national 0...
    $n = $norm;
    if (strpos($n, '+') === 0) $n = '00' . substr($n, 1);
    if (strpos($n, '00') === 0) {
        $digits = preg_replace('/\D+/', '', $n);
        if (preg_match('/^0031(\d+)/', $digits, $m)) {
            // Netherlands: drop 0031, add leading 0
            return '0' . $m[1];
        }
    }
    return null;
}

/** Country code check list: EU + also treat UK (44), USA/Canada (1), Turkey (90) as "Odido bundle mention" group */
function kcs_is_odido_bundle_cc($cc) {
    $cc = (int)$cc;
    $eu = array(
        30,31,32,33,34,36,39,40,43,45,46,48,49,
        351,352,353,356,357,358,359,
        370,371,372,
        385,386,
        420,421
    );
    if (in_array($cc, $eu, true)) return true;
    if ($cc === 44) return true;  // UK
    if ($cc === 1) return true;   // USA/Canada region
    if ($cc === 90) return true;  // Turkey
    return false;
}

/** Extract (best-effort) country code from a 00-prefixed digits string, e.g. 0032..., 0044..., 001... */
function kcs_extract_cc_from_00digits($digits) {
    // digits starts with 00
    $d = preg_replace('/\D+/', '', $digits);
    if (!preg_match('/^00(\d{1,3})/', $d, $m)) return null;

    // special: 001 is USA/Canada region and is valid
    // Just return first 1-3 digits after 00, but prefer known ones
    $rest = substr($d, 2);
    foreach (array(3,2,1) as $len) {
        if (strlen($rest) >= $len) {
            $cc = (int) substr($rest, 0, $len);
            if (kcs_is_odido_bundle_cc($cc)) return $cc;
        }
    }
    return (int) substr($rest, 0, min(3, strlen($rest)));
}

function kcs_analyze_number($raw) {
    $raw = trim((string)$raw);
    $norm = kcs_normalize($raw);

    // Fix: NL numbers entered as +31 / 0031 should be treated as NL national
    $nl_from_int = kcs_is_nl_mobile_or_fixed_from_international($norm);
    if ($nl_from_int) {
        $norm = $nl_from_int;
    }

    $digits = preg_replace('/\D+/', '', $norm);

    $type = 'Onbekend';
    $note = 'Controleer het nummer.';
    $extra = array();

    if ($norm === '' || $digits === '') {
        return array('normalized'=>'','digits'=>'','type'=>'Onbekend','note'=>'Vul een telefoonnummer in.','extra'=>array());
    }

    // Very short sequences
    if (strlen($digits) <= 2) {
        return array(
            'normalized'=>$norm,
            'digits'=>$digits,
            'type'=>'Nummerreeks',
            'note'=>'Dit lijkt op een nummerreeks, maar is nog niet compleet. Vul het volledige nummer in.',
            'extra'=>array('dtmf'=>kcs_to_dtmf_target($norm))
        );
    }

    // Handle + / 00 prefix series like 0032 (country code only)
    if (strpos($norm, '00') === 0 || strpos($norm, '+') === 0) {
        $dtmf = kcs_to_dtmf_target($norm);
        $d00 = (strpos($dtmf, '00') === 0) ? $dtmf : ('00' . ltrim($digits,'0'));

        // If only 00 + cc (like 0032 or 001 or 0044) with no subscriber number yet:
        if (preg_match('/^00\d{1,3}$/', $d00)) {
            $cc = kcs_extract_cc_from_00digits($d00);
            $type = 'Internationale landcode';
            $note = 'Dit is de internationale prefix (00) met een landcode. Vul daarna het volledige nummer in (bijv. 0032... voor België, 0044... voor het VK, 001... voor de USA).';
            $extra = array(
                'country_code' => $cc,
                'dtmf' => $d00
            );
            if ($cc !== null && kcs_is_odido_bundle_cc($cc)) {
                $extra['odido'] = true;
            }
            return array('normalized'=>$norm,'digits'=>$digits,'type'=>$type,'note'=>$note,'extra'=>$extra);
        }

        // Determine cc and mention Odido bundle if applicable
        $cc = kcs_extract_cc_from_00digits($d00);
        $type = 'Internationaal nummer';
        $note = 'Mogelijk (internationaal) tarief afhankelijk van je provider.';
        $extra = array('country_code'=>$cc, 'dtmf'=>$d00);
        if ($cc !== null && kcs_is_odido_bundle_cc($cc)) {
            $note .= ' Tip: met Odido onbeperkt (EU + USA + Turkije) bel je vaak voordelig.';
            $extra['odido'] = true;
        }
        return array('normalized'=>$norm,'digits'=>$digits,'type'=>$type,'note'=>$note,'extra'=>$extra);
    }

    // NL special
    if (preg_match('/^0800\d{4,}$/', $digits)) {
        $type = '0800-nummer (gratis)';
        $note = 'Dit is een 0800-nummer. In Nederland zijn 0800-nummers meestal gratis (check je provider).';
    } elseif (preg_match('/^0(900|909)\d{2,}$/', $digits)) {
        $type = '090x-nummer (betaald)';
        $note = 'Dit is een 090x-nummer. Dit kan extra belkosten hebben. Vul het tarief in je administratie zodra je het weet.';
    } elseif (preg_match('/^0(84|87)\d{2,}$/', $digits)) {
        $type = '084/087-nummer (mogelijk betaald)';
        $note = 'Dit is een 084/087-nummer. Dit kan een (verhoogd) tarief hebben, afhankelijk van je provider.';
    } elseif (preg_match('/^097\d{9}$/', $digits)) {
        $type = '097-nummer (M2M/data)';
        $note = 'Dit is een 097-nummer (vaak voor data/M2M). Tarief kan afwijken per provider.';
    } elseif (preg_match('/^06\d{8}$/', $digits)) {
        $type = 'Nederlands mobiel nummer';
        $note = 'Dit is een Nederlands mobiel nummer.';
    } elseif (preg_match('/^14\d{3}$/', $digits) || preg_match('/^140\d{2,3}$/', $digits)) {
        $type = '14+netnummer (overheid/bedrijf)';
        $note = 'Dit is een 14+netnummer. Tarief meestal lokaal, check je provider.';
    } elseif (preg_match('/^0\d{8,10}$/', $digits)) {
        $type = 'Nederlands nummer';
        $note = 'Dit lijkt een Nederlands vast/mobiel nummer.';
    } else {
        // Short or incomplete NL-like sequences
        if (preg_match('/^0\d{1,7}$/', $digits)) {
            $type = 'Nummerreeks';
            $note = 'Dit lijkt op een nummerreeks, maar is nog niet compleet. Vul het volledige nummer in.';
        } else {
            $type = 'Onbekend';
            $note = 'Dit nummerformaat herken ik niet helemaal. Probeer met landcode (+.. of 00..), of vul het volledige nummer in.';
        }
    }

    return array(
        'normalized'=>$norm,
        'digits'=>$digits,
        'type'=>$type,
        'note'=>$note,
        'extra'=>array('dtmf'=>kcs_to_dtmf_target($norm))
    );
}

/**
 * Extract candidates from text for autolink callout selection.
 * score: lower = preferred (cheaper).
 */
function kcs_extract_candidates($text, $blocked_digits = '') {
    if (!is_string($text) || $text === '') return array();

    $sep = '[\s\.\-\–\—\(\)]*';
    $patterns = array(
        '14'.$sep.'0\d{2,3}',
        '\\+(?:\\s*\\d'.$sep.'){6,14}\\d',
        '00(?:\\s*\\d'.$sep.'){2,14}\\d',
        '0(?:800|900|909)(?:'.$sep.'\\d){4,12}',
        '0(?:84|87)(?:'.$sep.'\\d){4,12}',
        '0(?:85|88)(?:'.$sep.'\\d){7,8}',
        '06(?:'.$sep.'\\d){8}',
        '097(?:'.$sep.'\\d){9}',
        '0(?:\\d'.$sep.'){6,10}\\d',
    );
    $master = '/(' . implode('|', $patterns) . ')/u';

    if (!preg_match_all($master, $text, $mm)) return array();

    $out = array();
    foreach ($mm[0] as $raw) {
        if (strpos($raw, '/') !== false) continue;
        $digits = preg_replace('/\\D+/', '', $raw);
        if (!$digits) continue;
        if ($blocked_digits && $digits === $blocked_digits) continue;

        $starts_plus = (strpos($raw, '+') !== false);
        $starts_00   = (bool) preg_match('/^\\s*00/u', $raw);

        if ($starts_plus) $target = '00' . ltrim($digits, '0');
        elseif ($starts_00) $target = $digits;
        else $target = $digits;

        $score = kcs_score_number($raw, $digits, $starts_plus, $starts_00);

        $out[] = array(
            'raw' => $raw,
            'digits' => $digits,
            'target' => $target,
            'score' => $score,
        );
    }

    return $out;
}

function kcs_score_number($raw, $digits, $starts_plus, $starts_00) {
    // Premium last
    if (preg_match('/^0\\s*9\\s*0\\s*(0|9)/u', $raw) || preg_match('/^09(00|09)/', $digits)) {
        return 60;
    }
    if (preg_match('/^0\\s*(84|87)/u', $raw) || preg_match('/^0(84|87)/', $digits)) {
        return 25;
    }
    if (preg_match('/^0\\s*8\\s*(5|8)/u', $raw) || preg_match('/^0(85|88)/', $digits)) {
        return 20;
    }
    if (preg_match('/^0\\s*8\\s*0\\s*0/u', $raw) || preg_match('/^0800/', $digits)) {
        return 30;
    }

    $is_international = $starts_plus || $starts_00;
    if ($is_international) {
        $dtmf = ($starts_plus) ? ('00' . ltrim($digits, '0')) : $digits;
        if (strpos($dtmf, '00') === 0) {
            $cc = kcs_extract_cc_from_00digits($dtmf);
            if ($cc !== null) {
                // prefer EU inside international bucket
                $eu = array(
                    30,31,32,33,34,36,39,40,43,45,46,48,49,
                    351,352,353,356,357,358,359,
                    370,371,372,
                    385,386,
                    420,421
                );
                if (in_array((int)$cc, $eu, true)) return 40;
            }
        }
        return 50;
    }

    return 10; // normal NL
}
