<?php
/*
Plugin Name: Klik & Check Suite (Bellen + Belkosten + AutoLink)
Description: Alles-in-één: nummer invoeren, belkosten/type checken, direct bellen of bel-instructies tonen. Autolinkt telefoonnummers in content naar tel:+31202621789,,<doel> en kan een belbalk met NL/UK/US + instructies tonen. Inclusief eenvoudige logging in wp-admin.
Version: 1.8.1
Author: Alexander van Dijl
License: GPL2+
*/

if (!defined('ABSPATH')) exit;

define('KCS_VER', '1.8.1');
define('KCS_TABLE', 'kcs_logs');

require_once __DIR__ . '/includes/helpers.php';

class KlikEnCheckSuite {

    const OUTCALL_NUMBER = '+31202621789'; // NL uitgaand
    const ACCESS_NL = '0202621789';
    const ACCESS_UK = '01284555131';
    const ACCESS_US = '17778728114';

    // Dit nummer nooit autolinken (in elke schrijfwijze met spaties/streepjes).
    const BLOCKED_LOCAL_NUMBER = '0202621789';

    private $skipTags = array('a','script','style','code','kbd','samp','pre','textarea','svg','math');

    public function __construct() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));

        // Shortcode widget
        add_shortcode('klik_en_check', array($this, 'shortcode_widget'));

        // AutoLink
        add_filter('the_content', array($this, 'filter_content'), 20);
        add_filter('the_excerpt', array($this, 'filter_content'), 20);

        // Ajax endpoints
        add_action('wp_ajax_nopriv_kcs_check', array($this, 'ajax_check'));
        add_action('wp_ajax_kcs_check', array($this, 'ajax_check'));
        add_action('wp_ajax_nopriv_kcs_log', array($this, 'ajax_log'));
        add_action('wp_ajax_kcs_log', array($this, 'ajax_log'));

        // Admin
        add_action('admin_menu', array($this, 'admin_menu'));
    }

    public static function activate() {
        global $wpdb;
        $table = $wpdb->prefix . KCS_TABLE;
        $charset_collate = $wpdb->get_charset_collate();

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $sql = "CREATE TABLE {$table} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            created_at DATETIME NOT NULL,
            event VARCHAR(32) NOT NULL,
            number_raw TEXT NULL,
            number_norm VARCHAR(64) NULL,
            number_hash CHAR(64) NULL,
            result_type VARCHAR(64) NULL,
            result_note TEXT NULL,
            page_url TEXT NULL,
            post_id BIGINT(20) UNSIGNED NULL,
            meta_json LONGTEXT NULL,
            PRIMARY KEY (id),
            KEY created_at (created_at),
            KEY event (event),
            KEY number_hash (number_hash)
        ) {$charset_collate};";
        dbDelta($sql);
    }

    public function enqueue_assets() {
        wp_enqueue_style('kcs-style', plugins_url('assets/kcs.css', __FILE__), array(), KCS_VER);
        wp_enqueue_script('kcs-script', plugins_url('assets/kcs.js', __FILE__), array(), KCS_VER, true);

        wp_localize_script('kcs-script', 'KCS', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('kcs_nonce'),

            'access_nl_label' => '020 262 1789',
            'access_uk_label' => '01284 555 131',
            'access_us_label' => '1-777-872-8114',

            'access_nl' => self::ACCESS_NL,
            'access_uk' => self::ACCESS_UK,
            'access_us' => self::ACCESS_US,
            'outcall'   => self::OUTCALL_NUMBER,

            'link_free_call' => 'https://alexandervandijl.nl/klik-om-te-bellen-plugin/',
            'link_odido'     => 'http://link.vraagalex.com/odido',

            'msg_empty' => 'Vul eerst een telefoonnummer in.',
            'msg_invalid' => 'Dat telefoonnummer lijkt niet te kloppen. Probeer het opnieuw.',
        ));
    }

    /** Shortcode: input + check + call buttons */
    public function shortcode_widget($atts) {
        $atts = shortcode_atts(array(
            'label' => 'Voer telefoonnummer in',
            'placeholder' => 'Bijv. 0900 123 45 67 of +32 2 123 45 67',
            'button_check' => 'Check tarief',
            'button_call' => 'Bel direct',
            'button_help' => 'Geef mij de instructies',
        ), $atts, 'klik_en_check');

        $id = 'kcs_' . wp_generate_uuid4();

        ob_start(); ?>
        <div class="kcs-card" data-kcs>
            <label class="kcs-label" for="<?php echo esc_attr($id); ?>"><?php echo esc_html($atts['label']); ?></label>
            <input id="<?php echo esc_attr($id); ?>" class="kcs-input" type="tel" inputmode="tel" autocomplete="tel" placeholder="<?php echo esc_attr($atts['placeholder']); ?>" />

            <div class="kcs-actions">
                <button type="button" class="kcs-btn kcs-btn-primary" data-kcs-action="check"><?php echo esc_html($atts['button_check']); ?></button>
                <button type="button" class="kcs-btn" data-kcs-action="call"><?php echo esc_html($atts['button_call']); ?></button>
                <button type="button" class="kcs-btn kcs-btn-outline" data-kcs-action="help"><?php echo esc_html($atts['button_help']); ?></button>
            </div>

            <div class="kcs-result" data-kcs-result hidden>
                <div class="kcs-row"><span class="kcs-k">Nummer</span><span class="kcs-v" data-kcs-number>—</span></div>
                <div class="kcs-row"><span class="kcs-k">Type</span><span class="kcs-v" data-kcs-type>—</span></div>
                <div class="kcs-row"><span class="kcs-k">Uitleg</span><span class="kcs-v" data-kcs-note>—</span></div>

                <div class="kcs-cta" data-kcs-cta hidden>
                    <div class="kcs-cta-title">Bellen (snelle opties)</div>
                    <div class="kcs-cta-buttons">
                        <a class="kcs-btn kcs-btn-primary" data-kcs-call="nl" href="#">🇳🇱 020 262 1789</a>
                        <a class="kcs-btn" data-kcs-call="uk" href="#">🇬🇧 01284 555 131</a>
                        <a class="kcs-btn" data-kcs-call="us" href="#">🇺🇸 1-777-872-8114</a>
                    </div>
                    <div class="kcs-cta-links">
                        <a href="<?php echo esc_url('https://alexandervandijl.nl/klik-om-te-bellen-plugin/'); ?>" target="_blank" rel="noopener">Gratis bellen via AlexandervanDijl.nl</a>
                        <span>·</span>
                        <a href="<?php echo esc_url('http://link.vraagalex.com/odido'); ?>" target="_blank" rel="noopener">Odido onbeperkt EU + USA + Turkije</a>
                    </div>
                </div>
            </div>

            <div class="kcs-help" data-kcs-help hidden>
                <div class="kcs-help-head">
                    <div class="kcs-help-title">📋 Bel-instructies</div>
                    <button type="button" class="kcs-help-close" data-kcs-close aria-label="Sluiten">✕</button>
                </div>
                <ol class="kcs-help-steps">
                    <li>Bel eerst één van deze toegangsnummers:
                        <ul>
                            <li><strong>Nederland:</strong> 020 262 1789</li>
                            <li><strong>Engeland:</strong> 01284 555 131</li>
                            <li><strong>USA:</strong> 1-777-872-8114</li>
                        </ul>
                    </li>
                    <li>Na verbinding toets je het gewenste nummer volledig in.</li>
                    <li><strong>Nederlandse nummers:</strong> kies alsof je vanuit Nederland belt (06 / 020 / 085 / 088 / 14… etc.).</li>
                    <li><strong>Buitenlandse nummers:</strong> altijd met <strong>00 + landcode</strong> (bijv. <strong>0032</strong>… voor België).</li>
                </ol>
                <div class="kcs-help-box">
                    <div class="kcs-row"><span class="kcs-k">Jij toetst:</span><span class="kcs-v" data-kcs-dial>—</span></div>
                </div>
            </div>

            <div class="kcs-msg" data-kcs-msg aria-live="polite"></div>
        </div>
        <?php
        return ob_get_clean();
    }

    /** AutoLink filter + optional callout */
    public function filter_content($html) {
        if (is_admin() || empty($html)) return $html;

        $linked = $this->autolink_html($html);

        // Append callout only on singular main content
        if (is_singular() && in_the_loop() && is_main_query()) {
            $text = wp_strip_all_tags($linked);
            $det = $this->detect_best_target($text);
            if (!empty($det['target'])) {
                $linked .= $this->render_callout($det);
            }
        }

        return $linked;
    }

    private function autolink_html($html) {
        if (stripos($html, '<') === false) {
            return $this->linkify_plain_text($html);
        }

        $doc = new DOMDocument('1.0', 'UTF-8');
        libxml_use_internal_errors(true);
        $wrapped = '<div id="__kcs_wrap__">'.$html.'</div>';
        @$doc->loadHTML(mb_convert_encoding($wrapped, 'HTML-ENTITIES', 'UTF-8'), LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        libxml_clear_errors();

        $xpath = new DOMXPath($doc);
        foreach ($xpath->query('//text()') as $textNode) {
            $parent = $textNode->parentNode;
            if (!$parent || in_array(strtolower($parent->nodeName), $this->skipTags, true)) continue;

            // Skip inside existing links
            for ($p = $parent; $p; $p = $p->parentNode) {
                if (strtolower($p->nodeName) === 'a') continue 2;
            }

            $orig = $textNode->nodeValue;
            $rep = $this->linkify_plain_text($orig);
            if ($rep !== $orig) {
                $frag = $doc->createDocumentFragment();
                // appendXML expects valid XML; our output is safe anchors
                @$frag->appendXML($rep);
                $parent->replaceChild($frag, $textNode);
            }
        }

        $out = '';
        $wrap = $doc->getElementById('__kcs_wrap__');
        if ($wrap) {
            foreach ($wrap->childNodes as $c) $out .= $doc->saveHTML($c);
        }
        return $out ?: $html;
    }

    private function linkify_plain_text($text) {
        if (!is_string($text) || $text === '') return $text;

        // IMPORTANT: no slash as separator (avoid 0800/0900)
        $sep = '[\s\.\-\–\—\(\)]*';

        $patterns = array(
            '14'.$sep.'0\d{2,3}',
            '\+(?:\s*\d'.$sep.'){6,14}\d',
            '00(?:\s*\d'.$sep.'){2,14}\d',  // allow short like 0032... but still needs digits later
            '0(?:800|900|909)(?:'.$sep.'\d){4,12}', // 0800 can be longer; 090x varies
            '0(?:84|87)(?:'.$sep.'\d){4,12}', // 084/087
            '0(?:85|88)(?:'.$sep.'\d){7,8}',
            '06(?:'.$sep.'\d){8}',
            '097(?:'.$sep.'\d){9}', // 097 + 9 digits = 12 total
            '0(?:\d'.$sep.'){6,10}\d', // general
        );

        $master = '/(' . implode('|', $patterns) . ')/u';

        return preg_replace_callback($master, function($m){
            $raw = $m[0];
            if (strpos($raw, '/') !== false) return $raw;

            $digits = preg_replace('/\D+/', '', $raw);
            if (!$digits) return $raw;

            // skip the access number on the site
            if ($digits === self::BLOCKED_LOCAL_NUMBER) return $raw;

            $starts_plus = (strpos($raw, '+') !== false);
            $starts_00 = (bool) preg_match('/^\s*00/u', $raw);

            if ($starts_plus) $target = '00' . ltrim($digits, '0');
            elseif ($starts_00) $target = $digits;
            else $target = $digits;

            $href = 'tel:' . self::OUTCALL_NUMBER . ',,' . $target;
            return sprintf('<a href="%s" class="kcs-tel-link">%s</a>', esc_attr($href), esc_html($raw));
        }, $text) ?? $text;
    }

    private function detect_best_target($text) {
        $cands = kcs_extract_candidates($text, self::BLOCKED_LOCAL_NUMBER);
        if (empty($cands)) return array('raw'=>null,'digits'=>null,'target'=>null,'score'=>null);

        $best = null;
        foreach ($cands as $c) {
            if ($best === null || (int)$c['score'] < (int)$best['score']) {
                $best = $c;
            }
        }
        return $best ?: array('raw'=>null,'digits'=>null,'target'=>null,'score'=>null);
    }

    private function render_callout($det) {
        $raw = $det['raw'];
        $target = $det['target'];

        $nl_href = 'tel:' . self::OUTCALL_NUMBER . ',,' . $target;
        $uk_href = 'tel:' . self::ACCESS_UK . ',,' . $target; // UK local access, no country code
        $us_href = 'tel:' . self::ACCESS_US . ',,' . $target; // US local access, no country code

        ob_start(); ?>
        <div class="kcs-callout" data-kcs-callout data-kcs-raw="<?php echo esc_attr($raw); ?>" data-kcs-target="<?php echo esc_attr($target); ?>">
            <div class="kcs-callout-title">Bel direct via (nummer ingevuld):</div>
            <div class="kcs-callout-sub">Gevonden nummer: <strong><?php echo esc_html($raw); ?></strong></div>
            <div class="kcs-callout-actions">
                <a class="kcs-btn kcs-btn-primary" data-kcs-callout-call="nl" href="<?php echo esc_attr($nl_href); ?>">🇳🇱 020 262 1789</a>
                <a class="kcs-btn" data-kcs-callout-call="uk" href="<?php echo esc_attr($uk_href); ?>">🇬🇧 01284 555 131</a>
                <a class="kcs-btn" data-kcs-callout-call="us" href="<?php echo esc_attr($us_href); ?>">🇺🇸 1-777-872-8114</a>
                <button type="button" class="kcs-btn kcs-btn-outline" data-kcs-callout-help>📋 Instructies</button>
            </div>
            <div class="kcs-callout-links">
                <a href="<?php echo esc_url('https://alexandervandijl.nl/klik-om-te-bellen-plugin/'); ?>" target="_blank" rel="noopener">Gratis bellen via AlexandervanDijl.nl</a>
                <span>·</span>
                <a href="<?php echo esc_url('http://link.vraagalex.com/odido'); ?>" target="_blank" rel="noopener">Odido onbeperkt EU + USA + Turkije</a>
            </div>

            <div class="kcs-help kcs-help-inline" data-kcs-help-inline hidden>
                <div class="kcs-help-head">
                    <div class="kcs-help-title">📋 Bel-instructies</div>
                    <button type="button" class="kcs-help-close" data-kcs-callout-close aria-label="Sluiten">✕</button>
                </div>
                <ol class="kcs-help-steps">
                    <li>Bel eerst één van deze toegangsnummers:
                        <ul>
                            <li><strong>Nederland:</strong> 020 262 1789</li>
                            <li><strong>Engeland:</strong> 01284 555 131</li>
                            <li><strong>USA:</strong> 1-777-872-8114</li>
                        </ul>
                    </li>
                    <li>Na verbinding toets je het gewenste nummer volledig in.</li>
                    <li><strong>Nederlandse nummers:</strong> kies alsof je vanuit Nederland belt.</li>
                    <li><strong>Buitenlandse nummers:</strong> altijd met <strong>00 + landcode</strong> (bijv. 0032… voor België).</li>
                </ol>
                <div class="kcs-help-box">
                    <div class="kcs-row"><span class="kcs-k">Wat jij moet intoetsen:</span><span class="kcs-v" data-kcs-callout-dial>—</span></div>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    /** AJAX: check number type + message */
    public function ajax_check() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field($_POST['nonce']), 'kcs_nonce')) {
            wp_send_json_error(array('message' => 'bad_nonce'), 403);
        }

        $raw = isset($_POST['number']) ? sanitize_text_field((string)$_POST['number']) : '';
        $raw = trim($raw);

        if ($raw === '') {
            wp_send_json_error(array('message' => 'empty'), 400);
        }

        $res = kcs_analyze_number($raw);

        // Log (privacy-friendly: hash + normalized)
        $this->insert_log(array(
            'event' => 'check',
            'number_raw' => $raw,
            'number_norm' => $res['normalized'],
            'number_hash' => hash('sha256', strtolower($res['normalized'])),
            'result_type' => $res['type'],
            'result_note' => $res['note'],
            'page_url' => isset($_POST['page_url']) ? esc_url_raw($_POST['page_url']) : null,
            'post_id' => isset($_POST['post_id']) ? absint($_POST['post_id']) : null,
            'meta_json' => wp_json_encode(array('extra' => $res['extra'])),
        ));

        wp_send_json_success($res);
    }

    /** AJAX: generic log for call clicks etc */
    public function ajax_log() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field($_POST['nonce']), 'kcs_nonce')) {
            wp_send_json_error(array('message' => 'bad_nonce'), 403);
        }

        $event = isset($_POST['event']) ? sanitize_text_field($_POST['event']) : '';
        $raw   = isset($_POST['number']) ? sanitize_text_field((string)$_POST['number']) : '';
        $norm  = kcs_normalize($raw);

        if ($event === '') $event = 'event';

        $this->insert_log(array(
            'event' => $event,
            'number_raw' => $raw ?: null,
            'number_norm' => $norm ?: null,
            'number_hash' => $norm ? hash('sha256', strtolower($norm)) : null,
            'page_url' => isset($_POST['page_url']) ? esc_url_raw($_POST['page_url']) : null,
            'post_id' => isset($_POST['post_id']) ? absint($_POST['post_id']) : null,
            'meta_json' => wp_json_encode(array(
                'access' => isset($_POST['access']) ? sanitize_text_field($_POST['access']) : null,
                'target' => isset($_POST['target']) ? sanitize_text_field($_POST['target']) : null,
                'raw_found' => isset($_POST['raw_found']) ? sanitize_text_field($_POST['raw_found']) : null,
            )),
        ));

        wp_send_json_success(array('ok'=>true));
    }

    private function insert_log($row) {
        global $wpdb;
        $table = $wpdb->prefix . KCS_TABLE;

        $data = array(
            'created_at' => current_time('mysql'),
            'event' => $row['event'],
            'number_raw' => $row['number_raw'],
            'number_norm' => $row['number_norm'],
            'number_hash' => $row['number_hash'],
            'result_type' => isset($row['result_type']) ? $row['result_type'] : null,
            'result_note' => isset($row['result_note']) ? $row['result_note'] : null,
            'page_url' => isset($row['page_url']) ? $row['page_url'] : null,
            'post_id' => isset($row['post_id']) ? $row['post_id'] : null,
            'meta_json' => isset($row['meta_json']) ? $row['meta_json'] : null,
        );

        $formats = array('%s','%s','%s','%s','%s','%s','%s','%s','%d','%s');
        $wpdb->insert($table, $data, $formats);
    }

    public function admin_menu() {
        add_management_page(
            'Klik & Check – Logs',
            'Klik & Check',
            'manage_options',
            'kcs-logs',
            array($this, 'render_admin_page')
        );
    }

    public function render_admin_page() {
        if (!current_user_can('manage_options')) return;

        global $wpdb;
        $table = $wpdb->prefix . KCS_TABLE;

        $rows = $wpdb->get_results("SELECT created_at,event,number_raw,result_type,page_url,meta_json FROM {$table} ORDER BY id DESC LIMIT 200", ARRAY_A);
        ?>
        <div class="wrap">
            <h1>Klik & Check – Logs</h1>
            <p>Laatste 200 events (checks en klik-logs). Tip: als je later tarieven wilt invullen voor 090x/084/087, kun je hier zien wat mensen invoeren.</p>

            <table class="widefat striped" style="max-width:1200px">
                <thead>
                    <tr>
                        <th>Datum</th>
                        <th>Event</th>
                        <th>Nummer</th>
                        <th>Type</th>
                        <th>Pagina</th>
                        <th>Meta</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($rows)): ?>
                        <tr><td colspan="6">Nog geen logs.</td></tr>
                    <?php else: foreach ($rows as $r): ?>
                        <tr>
                            <td><?php echo esc_html($r['created_at']); ?></td>
                            <td><?php echo esc_html($r['event']); ?></td>
                            <td><code><?php echo esc_html($r['number_raw']); ?></code></td>
                            <td><?php echo esc_html($r['result_type']); ?></td>
                            <td><?php echo esc_html($r['page_url']); ?></td>
                            <td><code style="white-space:pre-wrap;"><?php echo esc_html($r['meta_json']); ?></code></td>
                        </tr>
                    <?php endforeach; endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }
}

register_activation_hook(__FILE__, array('KlikEnCheckSuite', 'activate'));
new KlikEnCheckSuite();
